function [weights] = draw_importance(X, Y, method, feature_num, want_num)
% DRAW_IMPORTANCE Computes and visualizes the importance of each input variable
%   X         : Feature matrix
%   Y         : Class labels
%   method    : Naive Bayes method ('normal', 'mvmn', etc.)
%   feature_num : Total number of features
%   want_num  : Number of top important features to highlight
%   weights   : Importance score for each feature

    weights = zeros(1, feature_num);  % Initialize importance weights array

    for i = 1:feature_num
        % Train a Naive Bayes model using only the i-th feature
        if strcmp(method, 'mvmn')
            Mdl = fitcnb(X(:, i), Y, 'DistributionNames', method, 'CategoricalPredictors', "all");
        else
            Mdl = fitcnb(X(:, i), Y, 'DistributionNames', method);
        end

        % Compute the classification loss using only this feature
        L = loss(Mdl, X(:, i), Y);

        % Convert loss to an importance score: the smaller the loss, the higher the score
        weights(i) = min(10, -log(L));  % Cap score at 10
    end

    % Sort features by importance (descending order)
    [B, iranked] = sort(weights, 'descend');

    % Plot bar chart (horizontal) of importance values
    figure
    barh(weights(iranked), 'g');  % Green bars for all features
    xlabel('Variable Importance', 'FontSize', 26, 'Interpreter', 'latex');
    ylabel('Variable Rank', 'FontSize', 26, 'Interpreter', 'latex');
    title('Relative Importance of Inputs in estimating', 'FontSize', 15, 'Interpreter', 'latex');

    hold on

    % Highlight top `feature_num` features in yellow (optional, usually same as all)
    barh(weights(iranked(1:feature_num)), 'y');
    % Highlight top `want_num` important features in red
    barh(weights(iranked(1:want_num)), 'r');

    %--------------------------------------------------------------------------
    % Beautify and customize axes
    grid on
    xt = get(gca, 'XTick');
    xt_spacing = unique(diff(xt));  % Get spacing between X ticks
    xt_spacing = xt_spacing(1);
    yt = get(gca, 'YTick');
    ylim([0.25, length(weights) + 0.75]);  % Adjust Y-axis limits

    xl = xlim;
    xlim([0, 2.5 * max(weights)]);  % Extend X-axis to the right for labels

    % Add text labels to each bar, showing which column it came from
    for ii = 1:length(weights)
        text(...
            max([0 weights(iranked(ii)) + 0.02 * max(weights)]), ii, ...
            ['x ' num2str(iranked(ii))], 'Interpreter', 'latex', 'FontSize', 11);
    end

    %--------------------------------------------------------------------------
    % Set axes properties
    set(gca, 'FontSize', 16);
    set(gca, 'XTick', 0:2 * xt_spacing:1.1 * max(xl));
    set(gca, 'YTick', yt);
    set(gca, 'TickDir', 'out');
    set(gca, 'ydir', 'reverse');  % Flip Y-axis so highest importance is on top
    set(gca, 'LineWidth', 2);

    drawnow  % Ensure the figure is fully rendered

    % Save the figure
    root = find_root();  % Get root directory
    fn = fullfile(root, 'output_bayes', 'Variable_importance');  % Save path
    fnpng = [fn, '.png'];
    print('-dpng', fnpng);  % Save figure as PNG
    disp('Save completed')
    fprintf(' %s \n', fn);
end
